/*----------------------------------------------------------------------------
 * Name:    bc7701.c
 * Purpose:
 * Note(s):
 *----------------------------------------------------------------------------*/
#include	<string.h>
#include	"bc7701.h"
#include	"variable.h"
#include	"hci.h"

const tBCI_PACKAGE	BC7701_OperateSleep=
{
	BCI_CMD_PKG,				//head type
	3+1,							//length
	0x00,							//Flag
	BCI_OPERATE_MODE,			//opcode 
	OP_DEEPSLEEP				//parameter
};

const tBCI_PACKAGE	BC7701_OperateNormal=
{
	BCI_CMD_PKG,				//head type
	3+1,							//length
	0x00,							//Flag
	BCI_OPERATE_MODE,			//opcode 
	OP_NORMAL					//parameter
};

const tBCI_PACKAGE	BC7701_PowerDown=
{
	BCI_CMD_PKG,				//head type
	3+1,							//length
	0x00,							//Flag
	BCI_OPERATE_MODE,			//opcode 
	OP_POWERDOWN				//parameter
};

const tBCI_PACKAGE	BC7701_AdvertiseEnable=
{
	BCI_CMD_PKG,				//head type
	3+1,							//length
	0x00,							//Flag
	BCI_ADV_CTRL,				//opcode 
	ENABLE						//parameter	
};

const tBCI_PACKAGE	BC7701_AdvertiseDisable=
{
	BCI_CMD_PKG,				//head type
	3+1,							//length
	0x00,							//Flag
	BCI_ADV_CTRL,				//opcode 
	DISABLE						//parameter	
};

uc32	BaudRateTable[]={	9600,19200,25000,50000,62500,125000 };
#if (SYSTEM_CLOCK == FSYS16MHZ) 					/* for 16MHZ */
uc8	BaudRateTableHT8[]={103, 51, 39, 19, 15,  7};
#elif (SYSTEM_CLOCK == FSYS12MHZ) 				/* for 12MHZ */
uc8	BaudRateTableHT8[]={ 77, 38, 29, 14, 11,  5};
#elif (SYSTEM_CLOCK == FSYS8MHZ) 				/* for 8MHZ */
uc8	BaudRateTableHT8[]={ 51, 25, 19,  9,  7,  3};
#else 													/* for 4MHZ */    
uc8	BaudRateTableHT8[]={ 25, 12,  9,  4,  3,  1};
#endif

vu8	TransmitData[PACKAGE_MAX_SIZE];
vu8	ReciveData[PACKAGE_MAX_SIZE];

//vu8	uTxSerialBuffer[UART_BUFFER_SIZE];
vu8	uRxSerialBuffer[UART_BUFFER_SIZE];

//static u8	uTxWriteIdx,uTxReadIdx;
static u8	uRxWriteIdx,uRxReadIdx;
static u8	uPkgParserStep,uPkgLength,uPkgPosint;
static u8	uRxPkgType,uRxPkgEvent;

//#define IS_TXD_BUF_FULL(LEN)		(((uTxWriteIdx + LEN) >= UART_BUFFER_SIZE) ? ((uTxWriteIdx + LEN - UART_BUFFER_SIZE) == uTxReadIdx) : ((uTxWriteIdx + LEN) == uTxReadIdx))
//#define IS_TXD_BUF_EMPTY()      	(uTxReadIdx == uTxWriteIdx)
#define IS_RXD_BUF_FULL(LEN)		(((uRxWriteIdx + LEN) >= UART_BUFFER_SIZE) ? ((uRxWriteIdx + LEN - UART_BUFFER_SIZE) == uRxReadIdx) : ((uRxWriteIdx + LEN) == uRxReadIdx))
#define IS_RXD_BUF_EMPTY()      	(uRxReadIdx == uRxWriteIdx)
/*------------------------------------------------------------------------------
* Function Name : BC7701_InterfaceConfigure
* Description   : setup UART port & RESET pin for IO port
* Input         :
*				- br : UART Baud rate
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_InterfaceConfigure(u8 br)
{
	/* setup MCU I/O for BC7701 interface */
	/* PB1=TX2,PB2=RX2 */
	_pbs0 &= 0b11000011;
	_pbs0 |= 0b00101100;
	_rx2ps = LOW;	
	/* PD6 to I/O */
	_pds1 &= 0b11001111;
	_pdc6 = LOW;					/* output mode */
	_pd6 = LOW;	
	BC7701_UARTConfigure(br);
//	uTxWriteIdx = uTxReadIdx = 0;
	uRxWriteIdx = uRxReadIdx = 0;
	uPkgParserStep = PARS_PKG_TYPE;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_HardwareBaudRateDefault
* Description   : setup baud rate I/O status
* Input         : None
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/ 
void BC7701_HardwareBaudRateDefault(u8 br)
{
	BC77RXEN = DISABLE;			//RX disable
	BC77TXEN = DISABLE;			//TX disable	
	/* PB1/PB2 to I/O mode */
	_pbs0 &= 0b11000011;
	_pbc |= 0b00000110;		//MCU TXD-->BC7701 RXD=1 
	_pbpu |= 0b00000110;		//MCU RXD-->BC7701 TXD=1 
	/* For configure BC7701 baud rate*/
	/* BC7701 TXD=1 & RXD=1, baud rate = 115200 */
	/* BC7701 TXD=0 & RXD=1, baud rate = 9600 */	
	if(br == BAUD_RATE_9600)
	{
		_pbc2 = LOW;
		_pb2 = LOW;				//MCU RXD-->BC7701 TXD=0
	}	
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_HardwareBaudRateRelease
* Description   : TXD & RXD release to UART
* Input         : None
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_HardwareBaudRateRelease(void)
{
	/* PB1=TX2,PB2=RX2 */
	_pbs0 &= 0b11000011;
	_pbs0 |= 0b00101100;		
	_pbpu &= 0b11111001;	
	BC77INTF = FALSE;				//clear UART INT flag 	
	BC77RXEN = ENABLE;			//RX enable
	BC77TXEN = ENABLE;			//TX enable
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_UARTConfigure
* Description   : UART configure
*				- BaudRate = br
*				- Word Length = 8 Bits
*				- One Stop Bit
*				- None parity bit
* Input         :
*				- br : UART Baud rate
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_UARTConfigure(u8 br)
{	
	BC77UCR1 = 0b10000000;		//UARTEN=0,BNOn=0(8bits),PREN=0,STOPSn=0(1 stop bit)
	BC77UCR2	= 0b00100100;		//TXEN=0,RXEN=0,BRG=1(High Baud rate),RIE=1	
	BC77BRG = BaudRateTableHT8[br];		//set baud rate
	BC77RXEN = ENABLE;			//RX enable
	BC77TXEN = ENABLE;			//TX enable
	BC77INTF = FALSE;				//clear UART INT flag 
	BC77MIF = FALSE;				//clear MIF flag
	BC77INTE = ENABLE;			//enable UART INT
	BC77MIE = ENABLE;				//enable MF INT
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_UARTWakeUpCtrl
* Description   : MCU UART wake up function enable/disable
* Input         : 
*			  -ctrl: enable/disable UART wake up function
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_UARTWakeUpCtrl(u8 ctrl)
{	
	BC77WAKE = ctrl ? ENABLE : DISABLE;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SoftwareReset
* Description   : Reset command to BC7701
* Input         : None
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_SoftwareReset(void)
{
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = 7;
	BCIPKG(TransmitData)->flag = 0;	
	BCIPKG(TransmitData)->opcode = BCI_SOFT_RESET;
	*((u32 *)&BCIPKG(TransmitData)->param) = 0;
	BC7701_TransmitPackage(0);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SendHCIPackage
* Description   : send HCI command package to BC7701
* Input         : 
*				- opcode : HCI command
*				- len : parameter data length	
*				- pbuf : HCI parameter index
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_SendHCIPackage(u16 opcode,u8 len,u8 *pbuf)
{
	HCI_CMDPKG(TransmitData)->type = HCI_CMD_PKG;
	HCI_CMDPKG(TransmitData)->opcode = opcode;
	HCI_CMDPKG(TransmitData)->length = len;
   if((len != 0) && (pbuf != NULL))	memcpy(HCI_CMDPKG(TransmitData)->param,pbuf,len);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SendBCIPackage
* Description   : send BCI command package to BC7701
* Input         : 
*				- opcode : BCI command
*				- flag : BCI package flag
*				- len : parameter data length	
*				- pbuf : BCI parameter index
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_SendBCIPackage(u16 opcode,u8 flag,u8 len,u8 *pbuf)
{
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = len+3;		//flag+opcode=3byte
	BCIPKG(TransmitData)->flag = flag;
	BCIPKG(TransmitData)->opcode = opcode;	
   if((len != 0) && (pbuf != NULL))	memcpy(BCIPKG(TransmitData)->param,pbuf,len);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetDeviceName
* Description   : set BLE device name
* Input         : 
*				- leng : device name length(max 31byte)
*				- name : device name string
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_SetDeviceName(u8 leng,u8 *name)
{
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = leng+3;	// flag+opcode+parameter=leng+3 	
	BCIPKG(TransmitData)->flag = 0x00;
	BCIPKG(TransmitData)->opcode = BCI_DEV_NAME;
   if((leng != 0) && (name != NULL)) memcpy(BCIPKG(TransmitData)->param,name,leng);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetDeviceAddress
* Description   : set BLE device address(MAC address)
* Input         : 
*				- adr : device address(6 byte)
*				- type : device address type
*					- 0x00 : static address type
*					- 0x01 : random address type
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_SetDeviceAddress(u8 *adr,u8 type)
{
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = 7+3;	// flag+opcode+parameter=10 	
	BCIPKG(TransmitData)->flag = 0x00;
	BCIPKG(TransmitData)->opcode = BCI_DEV_ADDRESS;
   if(adr != NULL) memcpy(((tDEV_ADDR *)(BCIPKG(TransmitData)->param))->adr,adr,6);
	((tDEV_ADDR *)(BCIPKG(TransmitData)->param))->type = type;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_ConnectIntervalModify
* Description   : modify connect interval
* Input         : 
*				- opcode : BCI_CONN_INTV or BCI_CONN_INTV1
*				- min : Minimum time connections interval(unit 1.25ms)
*				- max : Maximum time connection interval(BCI_CONN_INTV1 only)(unit 1.25ms)	
* Output        : None
* Return        : 
*				- BCI_CONN_INTV opcode
*				- BCI_CONN_INTV1 opcode
------------------------------------------------------------------------------*/
void BC7701_ConnectIntervalModify(u16 opcode,u16 min,u16 max)
{
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->flag = 0x00;
	if(opcode == BCI_CONN_INTV)
	{
		BCIPKG(TransmitData)->opcode = BCI_CONN_INTV;
		BCIPKG(TransmitData)->length = 2+3;	// flag+opcode+parameter=5 
		((tCNNT_INTV *)(BCIPKG(TransmitData)->param))->min = min;		
	}
	else
	{
		BCIPKG(TransmitData)->opcode = BCI_CONN_INTV1;
		BCIPKG(TransmitData)->length = 8+3; // flag+opcode+parameter=11 
		((tCNNT_INTV *)(BCIPKG(TransmitData)->param))->min = min;				
		((tCNNT_INTV *)(BCIPKG(TransmitData)->param))->max = max;				
		((tCNNT_INTV *)(BCIPKG(TransmitData)->param))->latency = 0;				
		((tCNNT_INTV *)(BCIPKG(TransmitData)->param))->timeout = 300;	
	}
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetAdvertisingData
* Description   : set Advertising data
* Input         : 
*				- mode : How to set up Advertising data
*					- UPDAE_AUTO_NAME : Automatically add device name(DEV_Name command)
*					- UPDAE_NOAO_NAME : Do not automatically add device name
*					- DEFAULT_NAME : default Advertising data(flag + device name)
*					- DEFAULT_EMPTY : Set Advertising data to be empty
*				- leng : Advertising data length(max 31 bytes)
*				- advdata : advertising data
* Output        : None
* Return        : 
*				- TRUE : generate package ok
*				- FALSE : generate package fail,mode error
------------------------------------------------------------------------------*/
void BC7701_SetAdvertisingData(u8 mode,u8 leng,u8 *advdata)
{
	switch(mode)
	{
		case DEFAULT_NAME :
			BCIPKG(TransmitData)->length = 1+3;	// flag+opcode+parameter=4				
			BCIPKG(TransmitData)->flag = 0x00;			
			BCIPKG(TransmitData)->param[0] = 0x00;		
			break;
		case DEFAULT_EMPTY :
			BCIPKG(TransmitData)->length = 1+3;	// flag+opcode+parameter=4				
			BCIPKG(TransmitData)->flag = 0x00;			
			BCIPKG(TransmitData)->param[0] = 0xFF;
			break;
		case UPDAE_AUTO_NAME :
		case UPDAE_NOAO_NAME :		
		default :
			BCIPKG(TransmitData)->length = leng+3;	// flag+opcode+parameter=leng+3
			if(mode == UPDAE_NOAO_NAME)
				BCIPKG(TransmitData)->flag = UPDAE_NOAO_NAME;
			else
				BCIPKG(TransmitData)->flag = UPDAE_AUTO_NAME;			
			if((leng != 0) && (advdata != NULL)) 
				memcpy(BCIPKG(TransmitData)->param,advdata,leng);			
			break;			
	}
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->opcode = BCI_ADV_DATA;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetScanResponseData
* Description   : Set Scan Response Data
* Input         : 
*				- leng : Scan Response Data length(max 31 bytes)
*				- sdata : Scan Response Data
*				- Note : If leng=0 or sdata=NULL, 
*							Scan Response Data will be set to none
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_SetScanResponseData(u8 leng,u8 *sdata)
{
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = leng+3;	// flag+opcode+parameter=leng+3	
	BCIPKG(TransmitData)->flag = 0x00;	
	BCIPKG(TransmitData)->opcode = BCI_SCAN_DATA;
	if((leng != 0) && (sdata != NULL))
	{
		memcpy(BCIPKG(TransmitData)->param,sdata,leng);
	}
	else
	{
		BCIPKG(TransmitData)->length = 1+3;	// flag+opcode+parameter=4	
		BCIPKG(TransmitData)->param[0] = 0x00;
	}
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_AdvertisingInterval
* Description   : set Advertising interval to BC7701
* Input         : 
*				- min : Minimum time advertising interval(unit 0.625ms)
*				- max : Maximum time advertising interval(unit 0.625ms)
*				- chmap : advertising channel 
*					- b0 : channel 37(2402MHz)
*					- b1 : channel 38(2426MHz)
*					- b2 : channel 39(2480Mhz) 
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_AdvertisingInterval(u16 min,u16 max,u8 chmap)
{
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = 5+3;	// flag+opcode+parameter=8 	
	BCIPKG(TransmitData)->flag = 0x00;
	BCIPKG(TransmitData)->opcode = BCI_ADV_INTV;
	((tADV_INTV *)(BCIPKG(TransmitData)->param))->min = min;				
	((tADV_INTV *)(BCIPKG(TransmitData)->param))->max = max;				
	((tADV_INTV *)(BCIPKG(TransmitData)->param))->map = chmap & 0x07;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_AdvertisingControl
* Description   : control BC7701 enable or disable Advertising function
* Input         : 
*				- ctrl : ENABLE or DISABLE Advertising
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_AdvertisingControl(u8 ctrl)
{
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = 1+3;	// flag+opcode+parameter=4 	
	BCIPKG(TransmitData)->flag = 0x00;
	BCIPKG(TransmitData)->opcode = BCI_ADV_CTRL;
	if(ctrl == ENABLE)
		BCIPKG(TransmitData)->param[0] = 1;
	else
		BCIPKG(TransmitData)->param[0] = 0;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetTxPower
* Description   : set BC7701 Tx Power
* Input         : 
*				- pwr : Tx Power level 0x00~0x0F
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_SetTxPower(u8 pwr)
{
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = 1+3;	// flag+opcode+parameter=4
	BCIPKG(TransmitData)->flag = 0x00;
	BCIPKG(TransmitData)->opcode = BCI_TX_POWER;
	BCIPKG(TransmitData)->param[0] = pwr & 0x0F;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetCrystalCload
* Description   : set BC7701 crystal C-Load level
* Input         : 
*				- cc : crystal C-Load level 0x00~0x0F
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_SetCrystalCload(u8 cc)
{
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = 1+3;	// flag+opcode+parameter=4
	BCIPKG(TransmitData)->flag = 0x00;
	BCIPKG(TransmitData)->opcode = BCI_CLOAD_SET;
	BCIPKG(TransmitData)->param[0] = cc & 0x0F;
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetupFeatureFlag
* Description   : setup feature flag
* Input         : 
*				- md : update feature flag mode
*					-FEATURE_DIR : Direct update feature flag
*					-FEATURE_SET : set feature flag
*					-FEATURE_CLR : clear feature flag
*					-Others : no action
*				- sff : update feature flag data
* Output        : None
* Return        : 
*				- TRUE : generate package ok
*				- FALSE : generate package fail,md error
------------------------------------------------------------------------------*/
void BC7701_SetupFeatureFlag(u8 md,FeatureFlag sff)
{
	switch(md)
	{
		case FEATURE_DIR : md = FEATURE_DIR;	break;
		case FEATURE_CLR : md = FEATURE_CLR; sff = ~sff; break;
		case FEATURE_SET : 
		default :
			md = FEATURE_SET;	break;		
	}
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = 4+3;	// flag+opcode+parameter=7	
	BCIPKG(TransmitData)->flag = md;	
	BCIPKG(TransmitData)->opcode = BCI_FEATURE_CTRL;
	BCIPKG(TransmitData)->param[0]= ((u8 *)&sff)[0];
	BCIPKG(TransmitData)->param[1]= ((u8 *)&sff)[1];
	BCIPKG(TransmitData)->param[2]= ((u8 *)&sff)[2];
	BCIPKG(TransmitData)->param[3]= ((u8 *)&sff)[3];	
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetOperateMode
* Description   : set BC7701 operate mode
* Input         : 
*				- omd : operate mode
*					-OP_NORMAL : normal mode
*					-OP_DEEPSLEEP : deep sleep mode
*					-OP_POWERDOWN : power down mode(Only hardware reset)
*					-Others : No action
* Output        : None
* Return        : 
*				- TRUE : generate package ok
*				- FALSE : generate package fail,omd error
------------------------------------------------------------------------------*/
void BC7701_SetOperateMode(u8 omd,u8 wue,u8 wuw,u8 wut)
{
	switch(omd)
	{
		case OP_DEEPSLEEP :	omd = OP_DEEPSLEEP;	break;
		case OP_POWERDOWN :  omd = OP_POWERDOWN;	break;
		case OP_NORMAL :		
		default :		
			omd = OP_NORMAL;	break;		
	}
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->flag = 0x00;	
	BCIPKG(TransmitData)->opcode = BCI_OPERATE_MODE;
	BCIPKG(TransmitData)->param[0] = omd;			
	if(wue == ENABLE)
	{
		BCIPKG(TransmitData)->length = 8+3;	// flag+opcode+parameter=11
		BCIPKG(TransmitData)->param[1] = 0;
		BCIPKG(TransmitData)->param[2] = 0;
		BCIPKG(TransmitData)->param[3] = 0;
		BCIPKG(TransmitData)->param[4] = 1;
		BCIPKG(TransmitData)->param[5] = wut;
		BCIPKG(TransmitData)->param[6] = wuw;
		BCIPKG(TransmitData)->param[7] = 0;				
	}
	else
	{
		BCIPKG(TransmitData)->length = 1+3;	// flag+opcode+parameter=4	
	}
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetWhiteList
* Description   : set BC7701 white list for MAC address & erase before white list
* Input         : 
*				- erase : Enable or Disable erase before white list
*					-ENABLE : erase before white list
*					-DISABLE: no erase 
*				- adr : Add whitelisted address
*				- mask : Add the address mask of the whitelist
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_SetWhiteList(u8 erase,u8 *adr,u8 *mask)
{
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = 12+3;	// flag+opcode+parameter=15
	if(erase == ENABLE)
		BCIPKG(TransmitData)->flag = 0x10;	//erase before white list
	else
		BCIPKG(TransmitData)->flag = 0x00;			
	BCIPKG(TransmitData)->opcode = BCI_WHITE_LIST;
	memcpy(((tWHITE_LIST *)(BCIPKG(TransmitData)->param))->adr,adr,6);
	memcpy(((tWHITE_LIST *)(BCIPKG(TransmitData)->param))->mask,mask,6);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_SetBaudRate
* Description   : set BC7701 UART interface baud rate
* Input         : 
*				- br : baud rate type
*					- BAUD_RATE_4800 : baud rate 4800
*					- BAUD_RATE_9600 : baud rate 9600
*					- BAUD_RATE_19200 : baud rate 19200
*					- BAUD_RATE_25000 : baud rate 25000
*					- BAUD_RATE_50000 : baud rate 50000
*					- BAUD_RATE_62500 : baud rate 62500
*					- BAUD_RATE_115200 : baud rate 115200
* Output        : None
* Return        : 
*				- TRUE : generate package ok
*				- FALSE : generate package fail,baud rate error
------------------------------------------------------------------------------*/
void BC7701_SetBaudRate(u8 br)
{
	if(br > BAUD_RATE_115200) br=BAUD_RATE_115200;
	BCIPKG(TransmitData)->type = BCI_CMD_PKG;
	BCIPKG(TransmitData)->length = 4+3;	// flag+opcode+parameter=leng+3
	BCIPKG(TransmitData)->flag = 0x00;
	BCIPKG(TransmitData)->opcode = BCI_BAUDRATE;	
	memcpy(BCIPKG(TransmitData)->param,&BaudRateTable[br],sizeof(u32));
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_DummyWakeup
* Description   : send 0x00 data to wake up BC7701
* Input         : 
*				- len : send 0x00 data length	
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_DummyWakeup(u8 leng)
{
	while(leng)
	{
		while(!BC77TXIF) _nop();
		BC77TRXD = 0x00;
		leng--;
	}
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_TransmitCmdPackage
* Description   : send command only package data to BC7701 
* Input         :
*				- opcode : BCI command
*				- flag : BCI package flag
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_TransmitCmdPackage(u16 opcode,u8 flag)
{
	while(!BC77TXIF);
	BC77TRXD = BCI_CMD_PKG;				//BCI command packet head 0x77
	while(!BC77TXIF);
	BC77TRXD = 0x03;						//length=3byte
	while(!BC77TXIF);
	BC77TRXD = flag;						//flag byte
	while(!BC77TXIF);
	BC77TRXD = ((u8 *)&opcode)[0];	//opcode low byte
	while(!BC77TXIF);
	BC77TRXD = ((u8 *)&opcode)[1];	//opcode high byte		
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_TransmitPackage
* Description   : send package data to BC7701 
* Input         : 
*				- length : package length
*					- If it is an HCI & BCI package, use the length in the package
*				- pbuf : package index
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_TransmitPackage(u8 length)
{
	switch(TransmitData[0])
	{
		case HCI_CMD_PKG :
			length = HCI_CMDPKG(TransmitData)->length;
			/* type+opcode+length = 4byte */
			length += 4;
			break;
		case BCI_CMD_PKG :
			length = BCIPKG(TransmitData)->length;
			/* type+length = 2byte */
			length += 2;
			break;
	}
	GetOffsetLow(_mp1l,TransmitData);
	GetOffsetHigh(_mp1h,TransmitData);
	while(length)   
	{
		while(!BC77TXIF);
		BC77TRXD = _iar1;
		_mp1l++;
		length--;			
	}
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_TransmitPackageConst
* Description   : send package data to BC7701 
* Input         : 
*				- pkg : BCI package index
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BC7701_TransmitPackageConst(tBCI_PACKAGE *pkg)
{
	u8	length;
	u8	idx;
	
	length = pkg->length+2;	
	idx = 0;
   while(length)
   {
		while(!BC77TXIF);
		BC77TRXD = ((u8 *)pkg)[idx];
		idx++;
		length--;			
   }	
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_PackageParserProcess
* Description   : parser UART data to package
* Input         : None
* Output        : None
* Return        : 
* 				- bool flag
*					- TRUE : Parsing a packet is complete
*					- FALSE : Completed without parsing a packet
------------------------------------------------------------------------------*/
u8 BC7701_PackageParserProcess(void)
{
   u8	rdata;
   u8	pkgvalid;
	
   pkgvalid = FALSE;	
   while(!IS_RXD_BUF_EMPTY())
   {
		rdata = uRxSerialBuffer[uRxReadIdx];
		uRxReadIdx++;
		if(uRxReadIdx == UART_BUFFER_SIZE) uRxReadIdx = 0;
		switch(uPkgParserStep)
      {
         case PARS_EVENT_PARAM :
				ReciveData[uPkgPosint] = rdata;
				uPkgPosint++;
				if(--uPkgLength == 0)
            {
					pkgvalid = TRUE;
					uPkgPosint = 0;
               uPkgParserStep = PARS_PKG_TYPE;
            }
            break;
			case PARS_HCI_EVENT :
				ReciveData[uPkgPosint] = rdata;
				uPkgPosint++;
				uPkgParserStep = (rdata == EVENT_CMD_COMPLETE) ? PARS_EVENT_LENG : PARS_PKG_TYPE;
            break;
			case PARS_EVENT_LENG :
				ReciveData[uPkgPosint] = rdata;
				uPkgPosint++;
				uPkgLength = rdata;			
				//check package size if=0 or >max size to error
				switch(ReciveData[0])
				{
					case HCI_EVENT_PKG :             	/* 0x04 */
						if((uPkgLength < 4) || (uPkgLength > (PACKAGE_MAX_SIZE-2)))	
							uPkgParserStep = PARS_PKG_TYPE;
						else	
							uPkgParserStep = PARS_EVENT_PARAM;
						break;
					case BCI_EVENT_PKG :						/* 0x78 */
						if((uPkgLength < 3) || (uPkgLength > (PACKAGE_MAX_SIZE-2)))	
							uPkgParserStep = PARS_PKG_TYPE;
						else	
							uPkgParserStep = PARS_EVENT_OPCODE;
						break;
				}
            break;
			case PARS_EVENT_OPCODE :
				ReciveData[uPkgPosint] = rdata;
				uPkgPosint++;
				uPkgLength--;
				if(uPkgPosint == 5)
				{
					//check opcode 0x00XX,0xFFXX,0x18XX,0x2AXX 
					//0x00XX --> BCI Command
					//0xFFXX --> Unknow Service
					//0x18XX --> profile Service UUID
					//0x2AXX --> Characteristic UUID
					if((ReciveData[4] == 0x00) || (ReciveData[4] == 0xFF) ||
						(ReciveData[4] == 0x18) || (ReciveData[4] == 0x2A) 	)
					{
						if(uPkgLength > 0) uPkgParserStep = PARS_EVENT_PARAM;
						else
						{
							pkgvalid = TRUE;
							uPkgParserStep = PARS_PKG_TYPE;							
						}
					}
					else	uPkgParserStep = PARS_PKG_TYPE;
				}
				break;
			case PARS_PKG_TYPE :
            switch(rdata)
				{
					case HCI_EVENT_PKG :             	/* 0x04 */
                  ReciveData[0] = HCI_EVENT_PKG;
                  uPkgPosint = 1;
                  uPkgParserStep = PARS_HCI_EVENT; 	/* to receive event code data step */
                  break;
					case BCI_EVENT_PKG :						/* 0x78 */
                  ReciveData[0] = BCI_EVENT_PKG;
                  uPkgPosint = 1;
                  uPkgParserStep = PARS_EVENT_LENG; 	/* to receive length data step */
						break;
				}
            break;
		}
		if(pkgvalid) break;
   }
   return(pkgvalid);
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_UartISR
* Description   : UART interrupt process
* Input         : None
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void __attribute((interrupt(BC77ISRV))) BC7701_UartISR(void)
{	
	BC77INTF = LOW;				//clear UART INT flag 
	if(BC77USR & 0xB0)			// PERR & FERR & OERR
	{
		uRxSerialBuffer[uRxWriteIdx] = BC77TRXD;		
	}	
	if(BC77RXIF)
	{		
		uRxSerialBuffer[uRxWriteIdx] = BC77TRXD;
		if(++uRxWriteIdx == UART_BUFFER_SIZE)	uRxWriteIdx=0; 
	}
} 
 